/**
 *-----------------------------------------------------------------------------------
 *    Filename: ADFGenerate.c
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Copyright 2004-2007 Mitsubishi Electric Research Laboratories (MERL)
 *    An implementation for generating, destroying, and querying ADFs
 *    Ronald Perry and Sarah Frisken
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    Required include files for this implementation
 *-----------------------------------------------------------------------------------
 */
#include <stdlib.h>

#include "fs_object.h"
#include "fs_function.h"

/**
 *-----------------------------------------------------------------------------------
 *    START: iType Edge Rendering
 *-----------------------------------------------------------------------------------
 */
#ifdef FS_EDGE_RENDER

#include "adfgenerate.h"
#include "adfimplicit.h"
#include "adffixedmath.h"
#include "adfinittermsystem.h"

/**
 *-----------------------------------------------------------------------------------
 *    DRIVER FOR ADF GENERATION
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Generate an ADF from the specified ADFPath using the specified generation
 *    attributes. This function returns an opaque pointer to the generated ADF; a NULL
 *    is returned if the request cannot be satisfied.
 *-----------------------------------------------------------------------------------
 */
ADF_Void *ADFGenerateADF (void *libInst, ADFPath *path)
{


    /**
     *----Return NULL if the ADFPath is invalid
     */
    if (path == 0) return(0);


    /**
     *----Generate an implicit ADF
     */
    {
        /**
         *----Generate and return an implicit ADF
         */
        ADFGlyph *adf = ADFGenerateADFImplicit(libInst, path);
        return((void *) adf);
    }

}


/**
 *-----------------------------------------------------------------------------------
 *    ADF DESTRUCTION
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Destroy the given ADF
 *-----------------------------------------------------------------------------------
 */
ADF_Void ADFDestroyADF (void *libInst, void *ADF)
{
    if (ADF) ADF_FREE(ADFGetAppInst(libInst), ADF);
}


/**
 *-----------------------------------------------------------------------------------
 *    QUERYING THE ADF SIZE
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    The representation of an ADF is opaque (i.e., a void * pointer) and not
 *    accessible to the application. Applications can, however, query the ADF for its
 *    size, thus enabling the ADF to be copied, cached, transmitted, etc., for various
 *    application specific uses. ADFs are represented as a contiguous block of memory
 *    beginning at the opaque pointer. All internal ADF references (e.g., to ADF cells)
 *    are stored as offsets, rather than direct pointers, thereby allowing applications
 *    to freely move ADFs throughout their system.
 *-----------------------------------------------------------------------------------
 */
ADF_U32 ADFGetADFSize (void *ADF)
{
    ADFGlyph *adf = (ADFGlyph *) ADF;

    if (adf == 0) return(0);
    return(adf->totalSizeBytes);
}


/**
 *-----------------------------------------------------------------------------------
 *    SCALING AND OFFSETTING AN ADF PATH FROM FONT UNITS TO ADF COORDINATES
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Determine the scale and offset for transforming a glyph represented by a
 *    specified ADFPath from font units to the [0.0,1.0] x [0.0,1.0] ADF coordinate
 *    system and compute the EM box size of the typeface of the glyph in ADF units.
 *    ADFSetGlyphScaleAndOffset() allows the identical (i.e., bit-exact) computation of
 *    its output attributes during ADFGenerateADF() and ADFRenderSetupFromPath().
 *-----------------------------------------------------------------------------------
 */
ADF_Void ADFSetGlyphScaleAndOffset (ADFPath *path, ADF_F32 *FUToADFScale, 
                                    ADF_F32 *glyphOffsetX, ADF_F32 *glyphOffsetY, 
                                    ADF_F32 *ADFUnitsPerEM)
{
    {
        /**
         *----Fixed point math implementation
         */
        ADF_I1616 xOffsetOutline, yOffsetOutline;
        ADF_I1616 fontUnitsToADFScale;
        ADF_I1616 shrinkFactorInv = 0x00016666;
        ADF_I1616 width, height;
        ADF_I1616 maxExtent;
        ADF_I32      divStatus;


        /**
         *----Determine the width, height, and maximum extent of the bounding box of
         *----the glyph in font units
         */
        ADF_I1616 glyphMinX = FLOAT_TO_I1616(path->glyphMinX);
        ADF_I1616 glyphMaxX = FLOAT_TO_I1616(path->glyphMaxX);
        ADF_I1616 glyphMinY = FLOAT_TO_I1616(path->glyphMinY);
        ADF_I1616 glyphMaxY = FLOAT_TO_I1616(path->glyphMaxY);


        /**
         *----If this is a uniform-width stroke-based glyph, correct the glyph
         *----bounding box to account for the stroke width
         */
        if (path->pathType == ADF_UNIFORM_STROKE_PATH) {
            ADF_I1616 strokeRadius = FLOAT_TO_I1616(path->pathWidth) >> 1;
            glyphMinX -= strokeRadius;
            glyphMinY -= strokeRadius;
            glyphMaxX += strokeRadius;
            glyphMaxY += strokeRadius;
        }


        /**
         *----Determine the width, height, and maximum extent of the bounding box of
         *----the glyph in font units
         */
        width  = glyphMaxX - glyphMinX;
        height = glyphMaxY - glyphMinY;
        maxExtent = (width > height) ? width : height;
        if (maxExtent == 0) maxExtent = I1616_CONST_1;


        /**
         *----Scale the maximum extent of the glyph so that the glyph will lie well
         *----within the boundaries of the ADF coordinate system when it is scaled 
         *----and offset
         */
        maxExtent = I1616_MUL(maxExtent, shrinkFactorInv);

    
        /**
         *----Determine the x and y offsets which will place the center of the glyph
         *----at the center of a bounding square with sides of length maxExtent
         */
        xOffsetOutline = ((maxExtent - width)  >> 1) - glyphMinX;
        yOffsetOutline = ((maxExtent - height) >> 1) - glyphMinY;


        /**
         *----Determine the scale factor which will scale the bounding square to the
         *----[0.0,1.0] x [0.0,1.0] ADF coordinate system
         */
        fontUnitsToADFScale = I1616_DIV(I1616_CONST_1, maxExtent, &divStatus);
        *FUToADFScale = I1616_TO_FLOAT(fontUnitsToADFScale);


        /**
         *----Scale the glyph offset from font units to ADF coordinates. Note that
         *----the glyph origin (0,0) in font units maps to
         *----(glyphOffsetX,glyphOffsetY) in ADF coordinates.
         */
        *glyphOffsetX = I1616_TO_FLOAT(I1616_MUL(
        xOffsetOutline, fontUnitsToADFScale));
        *glyphOffsetY = I1616_TO_FLOAT(I1616_MUL(
        yOffsetOutline, fontUnitsToADFScale));
    

        /**
         *----Determine the EM box size of the typeface of the glyph in ADF units
         */
        *ADFUnitsPerEM = I1616_TO_FLOAT(I1616_MUL(FLOAT_TO_I1616(
        path->fontUnitsPerEM), fontUnitsToADFScale));
    }
}

/**
 *-----------------------------------------------------------------------------------
 *    END: iType Edge Rendering
 *-----------------------------------------------------------------------------------
 */
#endif
